#include "libfma.h"
#include "lf_internal.h"
#include "lf_fabric.h"
#include "lf_process_fabric.h"
#include "lf_xbar32.h"

/*
 * local prototypes
 */
static void lf_propagate_subfabric_id(struct lf_nic *rnicp, int rport);

/*
 * Assign a sub-fabric ID to each NIC.  sub_fabric IDs are all
 * powers of 2 so that we can or them toegether to create a subfabric
 * mask for the owning host.
 */
void
lf_assign_subfabrics(
  struct lf_fabric *fp)
{
  struct lf_host *hp;
  struct lf_nic *nicp;
  int subfabric_id;
  int subfabric_mask;
  int h;
  int n;
  int p;

  subfabric_id = 1;

  for (h=0; h<fp->num_hosts; ++h) {
    hp = fp->hosts[h];
    subfabric_mask = 0;
    for (n=0; n<hp->num_nics; ++n) {
      nicp = hp->nics[n];
      for (p=0; p<nicp->num_ports; ++p) {
	if (FP_NIC(nicp)->subfabric_id[p] == 0 && nicp->topo_ports[p] != NULL) {
	  FP_NIC(nicp)->subfabric_id[p] = subfabric_id;
	  lf_propagate_subfabric_id(nicp, p);
	  subfabric_id <<= 1;
	}

	subfabric_mask |= FP_NIC(nicp)->subfabric_id[p];
      }
    }

    /* host mask is OR of all NIC subfabric IDs */
    hp->subfabric_mask = subfabric_mask;
  }
}

/*
 * Propagate subfabric ID out from a NIC port
 */
static void
lf_propagate_subfabric_id(
  struct lf_nic *rnicp,
  int rport)
{
  struct lf_xbar *work_list;
  struct lf_xbar *new_list;
  union lf_node *np;
  struct lf_fp_xbar *fpxp;
  struct lf_fp_xbar *nfpxp;
  struct lf_xbar *xp;
  int subfabric_id;
  int oport;
  int p;

  subfabric_id = FP_NIC(rnicp)->subfabric_id[rport];

  np = rnicp->topo_ports[rport];
  oport = rnicp->topo_rports[rport];

  /* direct connect is simple, set and return */
  if (np->ln_type == LF_NODE_NIC) {
    FP_NIC_N(np)->subfabric_id[oport] = subfabric_id;
    return;
  }

  /* Put this xbar alone on the work list */
  fpxp = FP_XBAR_N(np);

  fpxp->bf_next = NULL;
  work_list = LF_XBAR(np);
  fpxp->bf_last_visitor = rnicp;
  fpxp->bf_in_port = oport;

  /* Work the list BFS */
  new_list = NULL;

  while (work_list != NULL) {

    /* take the top item off the work list */
    xp = work_list;
    work_list = FP_XBAR(xp)->bf_next;

    /* Check each port for an xbar */
    for (p=0; p<xp->num_ports; ++p) {

      fpxp = FP_XBAR(xp);

      /* don't go out how we came in */
      if (p == fpxp->bf_in_port) continue;

      /* obey quadrant disable */
      if (lf_xbar_qd_violation(xp, fpxp->bf_in_port, p)) continue;

      /* If this link is down, skip it */
      if (xp->link_state[p] != LF_LINK_STATE_UP) continue;

      /* If this is an xbar not yet distanced, process it! */
      np = xp->topo_ports[p];
      oport = xp->topo_rports[p];
      if (np == NULL) continue;

      if (np->ln_type == LF_NODE_XBAR) {

	nfpxp = FP_XBAR_N(np);

	/* skip already visited xbars */
	if (nfpxp->bf_last_visitor == rnicp) continue;

	/* Add this xbar to the work list */
	nfpxp->bf_next = new_list;
	new_list = LF_XBAR(np);

	nfpxp->bf_in_port = oport;
	nfpxp->bf_last_visitor = rnicp;

      /* found another NIC for this subfabric */
      } else if (np->ln_type == LF_NODE_NIC) {
	FP_NIC_N(np)->subfabric_id[oport] = subfabric_id;
      }
    }

    /* If the list is empty, switch to new list */
    if (work_list == NULL) {
      work_list = new_list;
      new_list = NULL;
    }
  }
}

